
/**
 ******************************************************************************
 *
 * @file        MG32_USBD_Audio_Core.h
 * @brief       This file contains all the functions prototypes for the USB audio 
 *              core firmware library.
 *
 * @par         Project
 *              MG32
 * @version     V1.02
 * @date        2022/09/26
 * @author      Megawin Software Center
 * @copyright   Copyright (c) 2017 MegaWin Technology Co., Ltd.
 *              All rights reserved.
 *
 ******************************************************************************* 
 * @par Disclaimer
 * The Demo software is provided "AS IS" without any warranty, either
 * expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose. The author will
 * not be liable for any special, incidental, consequential or indirect
 * damages due to loss of data or any other reason.
 * These statements agree with the world wide and local dictated laws about
 * authorship and violence against these laws.
 *******************************************************************************
 *******************************************************************************
 */


#ifndef _USB_AUDIO_CORE_H
#define _USB_AUDIO_CORE_H


/* Includes ------------------------------------------------------------------*/
#include "MG32.h"
#include "MG32_USB.h"
#include "MG32__ExtraStruct.h"

/* Wizard menu ---------------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
//==========================================================
// Audio Relationship Informace
#define USB_IN_ID                               1
#define AUDIO_IN_ID                             2
#define USB_OUT_ID                              7
#define AUDIO_OUT_ID                            6
#define SELECTOR_ID                             8
#define FEATURE1_ID                             9
#define FEATURE2_ID                             10
                            
#define OUT_ENDPOINT_REG                        USB_EP1
#define OUT_ENDPOINT_ID                         1
#define OUT_tLowSamFreq                         8000
#define OUT_tCURSamFreq                         8000
#define OUT_tUpperSamFreq                       16000
#define OUT_BitResolution                       16
#define OUT_Channel                             2
                            
#define IN_ENDPOINT_REG                         USB_EP2
#define IN_ENDPOINT_ID                          2
#define IN_tLowSamFreq                          8000
#define IN_tCURSamFreq                          8000
#define IN_tUpperSamFreq                        16000
#define IN_BitResolution                        16
#define IN_Channel                              2
                            
#define USB_MULTIMEDIA_ENDPOINT_REG             USB_EP7
#define USB_MULTIMEDIA_AUDIOOUT_ID              1
#define USB_MULTIMEDIA_AUDIOIN_ID               2
#define USB_MULTIMEDIA_DATABYTE                 1
                            
#define AUDIO_OUT_FEATURE_ID                    FEATURE1_ID
#define AUDIO_IN_FEATURE_ID                     FEATURE2_ID

//===========================================================
#define AUDIO_SAM_FREQ_BYTE0(__FREQ__)          (( __FREQ__ & 0x0000FF) >> 0)
#define AUDIO_SAM_FREQ_BYTE1(__FREQ__)          (( __FREQ__ & 0x00FF00) >> 8)
#define AUDIO_SAM_FREQ_BYTE2(__FREQ__)          (( __FREQ__ & 0xFF0000) >> 16)
        
#define AUDIO_EP_SIZE( __FREQ__, __BITRES__, __CH__)  ((((__FREQ__) / 1000) * ((__BITRES__)/8)) * (__CH__))
#define AUDIO_EP_SIZE2( __FREQ__, __BITRES__, __CH__) (((((__FREQ__) / 1000) * ((__BITRES__)/8)) * (__CH__)) + 1)

//===========================================================
//User Define (For internal flow)
#define AUDIO_CONTROL                           0x43
#define AUDIO_ENDPOINT                          0x48
                                            
#define AUDIO_DATA_CUR                          0
#define AUDIO_DATA_MIN                          1
#define AUDIO_DATA_MAX                          2
#define AUDIO_DATA_RES                          3

//===========================================================
//Audio Request
//------------------------------------------------------
//bmRequestType
#define USB_AUDIO_REQUEST_TYPE_MASK             0x3F
#define USB_AUDIO_REQUEST_TYPE_ENDPOINT         0x22

//------------------------------------------------------
//bRequest
// CUR : Current setting attribute
// MIN : Minimum setting attribute
// MAX : Maximum setting attribute
// RES : Resolution attribute
// MEM : Memory space attribute
#define USB_AUDIO_REQUEST_UNDEFINED             0x00
#define USB_AUDIO_REQUEST_SET_CUR               0x01
#define USB_AUDIO_REQUEST_SET_MIN               0x02
#define USB_AUDIO_REQUEST_SET_MAX               0x03
#define USB_AUDIO_REQUEST_SET_RES               0x04
#define USB_AUDIO_REQUEST_SET_MEM               0x05
                                    
#define USB_AUDIO_REQUEST_GET_CUR               0x81
#define USB_AUDIO_REQUEST_GET_MIN               0x82
#define USB_AUDIO_REQUEST_GET_MAX               0x83
#define USB_AUDIO_REQUEST_GET_RES               0x84
#define USB_AUDIO_REQUEST_GET_MEM               0x85
#define USB_AUDIO_REQUEST_GET_STAT              0xFF
        
//===========================================================
#define CS_INTERFACE_DESCRIPTOR_TOTAL_SIZE      111
        
#define CS_HEADER_SIZE                          10
                    
#define CS_INPUT_TERMINAL_SIZE                  12
#define CS_OUTPUT_TERMINAL_SIZE                 19
#define CS_CLOCK_SOURCE_SIZE                    12
#define CS_POWER_DOMAIN_SIZE                    13
                    
#define AS_CLASS_INTERFACE_SIZE                 23
#define AS_CLASS_VALID_FREQ_INTERFACE_SIZE      11
//-----------------------------------------------------------
#define AS_INTERFACE_SUBTYPE_UNDEFINED          0x00
#define AS_INTERFACE_SUBTYPE_GENERAL            0x01
#define AS_INTERFACE_VALID_FREQ_RANGE           0x02
        
#define AUDIO_INTERFACE_CLASS_AUDIO             0x01              

#define AUDIO_INTERFACE_SUCLASS_UNDEFINE        0x00
#define AUDIO_INTERFACE_SUCLASS_AUDIOCONTROL    0x01
#define AUDIO_INTERFACE_SUCLASS_AUDIOSTREAMING  0x02
#define AUDIO_INTERFACE_SUCLASS_MIDISTREAMING   0x03
        
#define AUDIO_INTERFACE_PROTOCOL_IP_VER_01_00   0x00
#define AUDIO_INTERFACE_PROTOCOL_IP_VER_02_00   0x20
#define AUDIO_INTERFACE_PROTOCOL_IP_VER_03_00   0x30
        
#define CS_DEVICE_DESCRIPTOR                    0x21
#define CS_CONFIGURATION_DESCRIPTOR             0x22
#define CS_STRING_DESCRIPTOR                    0x23
#define CS_INTERFACE_DESCRIPTOR                 0x24
#define CS_ENDPOINT_DESCRIPTOR                  0x25
#define CS_CLUSTER_DESCRIPTOR                   0x26
        
#define CS_INTERFACE_SUBTYPE_UNDEFINED          0x00
#define CS_INTERFACE_SUBTYPE_HEADER             0x01
#define CS_INTERFACE_SUBTYPE_INPUT_TERMINAL     0x02
#define CS_INTERFACE_SUBTYPE_OUTPUT_TERMINAL    0x03
#define CS_INTERFACE_SUBTYPE_MIXER_UNIT         0x04
#define CS_INTERFACE_SUBTYPE_SELECTOR_UNIT      0x05
#define CS_INTERFACE_SUBTYPE_FEATURE_UNIT       0x06
#define CS_INTERFACE_SUBTYPE_PROCESSING_UINT    0x07
#define CS_INTERFACE_SUBTYPE_EXTENSION_UNIT     0x08

//===========================================================
// USB Device Class Definition for Terminal Types (Release 1.0)
//     2. Terminal Types
//         2.1 USB Terminal Type
//         2.2 Input Terminal Type
//         2.3 Ouput Terminal Type
//         2.4 Bi-directional Terminal Types
#define USB_TERMINAL_TYPE_UNDEFINE              0x0100    // I / O
#define USB_TERMINAL_TYPE_STREAMING             0x0101    // I / O
#define USB_TERMINAL_TYPE_VENDOR                0x01FF    // I / O
                                                             
#define INPUT_TERMINAL_TYPE_UNDEFINED           0x0200    // I
#define INPUT_TERMINAL_TYPE_MICROPHONE          0x0201    // I
#define INPUT_TERMINAL_TYPE_DESKTOP_MICROPHONE  0x0202    // I
#define INPUT_TERMINAL_TYPE_PERSONAL_MICROPHONE 0x0203    // I
#define INPUT_TERMINAL_TYPE_OMNI_MICROPHONE     0x0204    // I
#define INPUT_TERMINAL_TYPE_MICROPHONE_ARRAY    0x0205    // I
#define INPUT_TERMINAL_TYPE_PROCESSING_MICROPHINE_ARRAY 0x0206    // I
        
#define OUTPUT_TERMINAL_TYPE_UNDEFINED          0x0300    // O
#define OUTPUT_TERMINAL_TYPE_SPEAKER            0x0301    // O
#define OUTPUT_TERMINAL_TYPE_HEADPHINES         0x0302    // O
#define OUTPUT_TERMINAL_TYPE_HEAD_MOUNTED_DISPLAY_AUDIO 0x0303    // O
#define OUTPUT_TERMINAL_TYPE_DESKTOP_SPEAKER    0x0304    // O
#define OUTPUT_TERMINAL_TYPE_ROOM_SPEAKER       0x0305    // O
#define OUTPUT_TERMINAL_TYPE_COMMUNICATION_SPEAKER  0x0306    // O
#define OUTPUT_TERMINAL_TYPE_LOW_FREQUENCY_EFFECTS_SPEAKER  0x0307    // O
               
#define BI_DIR_TERMINAL_TYPE_UNDEFINED          0x0400    // I/O
#define BI_DIR_TERMINAL_TYPE_HANDSET            0x0401    // I/O
#define BI_DIR_TERMINAL_TYPE_HEADSET            0x0402    // I/O
#define BI_DIR_TERMINAL_TYPE_SPK_PHONE_NO_ECHO_REDUCTION  0x0403    // I/O
#define BI_DIR_TERMINAL_TYPE_ECHO_SUPPRESSING_SPK_PHONE   0x0404    // I/O
#define BI_DIR_TERMINAL_TYPE_ECHO_CANCELING_SPK_PHONE     0x0405    // I/O

//==========================================================
// USB Device Class Definition for Audio Device (Release 1.0)
//     * A.10.2 Feature Unit Control Selectors
#define FU_CONTROL_UNDEFINED                    0x00
#define FU_CONTROL_MUTE                         0x01
#define FU_CONTROL_VOLUME                       0x02
#define FU_CONTROL_BASS                         0x03
#define FU_CONTROL_MID                          0x04
#define FU_CONTROL_TREBLE                       0x05
#define FU_CONTROL_GRAPHIC_EQUALIZER            0x06
#define FU_CONTROL_AUTOMATIC_GAIN               0x07
#define FU_CONTROL_DELAY                        0x08
#define FU_CONTROL_BASS_BOOST                   0x09
#define FU_CONTROL_LOUDNESS                     0x0A

//============================================================
// Audio Endpoint Control Selectors
#define EP_CONTROL_UNDEFINED                    0x00
#define SAMPLING_FREQ_CONTROL                   0x01
#define PITCH_CONTROL                           0x02

//===========================================================
//
/**
 * @name  AUDIO_MULTIMEDIA_TypeDef : 
 *   		
 */ 
///@{
#define AUDIO_MULTIMEDIA_AUDIO_TYPE_MASK        0x8000
#define AUDIO_MULTIMEDIA_AUDIO_OUT              0x0000
#define AUDIO_MULTIMEDIA_AUDIO_IN               0x8000
///@}

/* Private typedef -----------------------------------------------------------*/
typedef enum
{
    AUDIO_MULTIMEDIA_MAKE   = 0x00,
    AUDIO_MULTIMEDIA_BREAK  = 0x01,
}MultimediaStatus_TypeDef;
    
typedef enum
{
    AUDIO_MULTIMEDIA_IDLE              = 0x0000,    /*!< Audio OUT Function */
    AUDIO_MULTIMEDIA_VOLUME_UP         = 0x0001,    /*!< Audio OUT Function */
    AUDIO_MULTIMEDIA_VOLUME_DOWN       = 0x0002,    /*!< Audio OUT Function */
    AUDIO_MULTIMEDIA_PLAY_PAUSE        = 0x0004,    /*!< Audio OUT Function */
    AUDIO_MULTIMEDIA_NEXT_TRACK        = 0x0008,    /*!< Audio OUT Function */
    AUDIO_MULTIMEDIA_PREVIOUS_TRACK    = 0x0010,    /*!< Audio OUT Function */
    AUDIO_MULTIMEDIA_STOP              = 0x0020,    /*!< Audio OUT Function */
    AUDIO_MULTIMEDIA_MUTE              = 0x0040,    /*!< Audio OUT Function */
    AUDIO_MULTIMEDIA_HOOKSWITCH        = 0x8001,    /*!< Audio IN  Function */  
}MultimediaFunction_TypeDef;

//===========================================================
typedef struct
{ 
    //==============================================
    //USB control parmaeter Tmp
    uint16_t         IDS; 
    uint8_t          CS;
    uint8_t          CN;
    uint8_t          bRequest;
    uint8_t          IdleRate;
    //==============================================
    //Audio IN / OUT Control stauts information.
    //    * CUR = Current setting attribute
    //    * MIN = Minimum setting attribute
    //    * MAX = Maximum setting attribute
    //    * RES = Resolution attribute
    uint8_t          AudioOUT_MUTE;
    uint8_t          AudioIN_MUTE;
    uint16_t         AudioOUT_Volume[OUT_Channel][4];
    uint16_t         AudioIN_Volume[IN_Channel][4];

    //==============================================
    //Audio IN / OUT Endpoint status information
    uint32_t         AudioOUT_SampleFreq[4];
    uint8_t          AudioOUT_PitchEnable_CUR;
    uint8_t          AudioOUT_DataByte;
    
    uint8_t          AudioIN_PitchEnable_CUR;
    uint8_t          AudioIN_DataByte;
    uint32_t         AudioIN_SampleFreq[4];
    //==============================================
    //Audio IN / OUT Data Control Relationship
    uint8_t*         AudioOUT_Buf;
    uint8_t          AudioOUT_Busyflag;
    uint8_t          AudioOUT_Cnt;
    
    uint8_t          AudioIN_Busyflag;
    uint8_t          AudioIN_Cnt;
    uint8_t*         AudioIN_Buf;
    //==============================================
    //Audio Multmedia Control Relationship
    uint8_t          AudioMultimedia_Busyflag;
    uint8_t          AudioOUT_MultimediaStatus;
    uint8_t          AudioIN_MultimediaStatus;
    uint8_t          AudioMultimedia_Buf[2];
    //==============================================
    //Other
    uint8_t          Busy;                                  // Speak Status
    uint8_t          __Reserved[2];
}USBAudio_TypeDef;

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
//===============================================================
void USB_Audio_Control_Request(uint8_t USBA_RequestType);
void USB_Audio_Endpoint_Request(void);

void USB_Audio_Control_SetRequest(void);
void USB_Audio_Endpoint_SetRequest(void);

//---------------------------------------------------------------
void USB_Audio_Inital(void);
void USB_Audio_OUTData(void);

void USB_Audio_INDataContinuous(void);
void USB_Audio_INDataTrigger(void);

void USB_Audio_TriggerMultimediaControl( MultimediaFunction_TypeDef mFunction , MultimediaStatus_TypeDef MStatus);

//---------------------------------------------------------------
void USB_Audio_InitCallback(void);

void USB_Audio_OUTDataCallback(uint8_t AudioData_Byte);

void USB_Audio_INDataTriggerCallback(void);
void USB_Audio_INDataContinuousCallback(void);

void USB_AudioOUT_UpdateMuteCallback(void);
void USB_AudioIN_UpdateMuteCallback(void);

void USB_AudioOUT_UpdateVolumeStatusCallback(void);
void USB_AudioIN_UpdateVolumeStatusCallback(void);

void USB_AudioOUT_UpdateSampleFrequenceCallback(void);
void USB_AudioIN_UpdateSampleFrequenceCallback(void);

/* Exported variables --------------------------------------------------------*/
extern __ALIGNED(2) uint8_t  TST_USB_AudioBuf[64];

/* Exported functions --------------------------------------------------------*/
/* External vairables --------------------------------------------------------*/
extern USBAudio_TypeDef USB_Audio;


#endif

















